import 'package:flutter/material.dart';
import 'package:flutter_riverpod/flutter_riverpod.dart';
import 'package:razinsoft_rider/core/extensions/extensions.dart';
import 'package:razinsoft_rider/core/widgets/is_ios.dart';
import 'package:razinsoft_rider/presentation/account_page/provider/theme_provider.dart';

import '../../theme/color_palette.dart';

class AppPrimaryButton extends ConsumerWidget {
  final Function()? onPressed;
  final Widget child;
  final bool isDisabled;
  final PrimaryButtonColor color;
  final Color? backgroundColor;
  final bool isLoading;
  final double width;
  final bool showBorder;

  const AppPrimaryButton({
    super.key,
    required this.onPressed,
    required this.child,
    this.isDisabled = false,
    this.isLoading = false,
    this.color = PrimaryButtonColor.primary,
    this.backgroundColor,
    this.width = double.infinity, this.showBorder = true,
  });

  @override
  Widget build(BuildContext context, WidgetRef ref) {
    final bool isDark = ref.read(themeModeProvider.notifier).isDarkMode();
    return SafeArea(
      bottom: !isIos(),
      child: ElevatedButton(
        onPressed: (isDisabled || isLoading) ? null : onPressed,
        style: ButtonStyle(
            minimumSize: WidgetStatePropertyAll(
              Size(width, 48),
            ),
            padding: WidgetStateProperty.all(
              const EdgeInsets.symmetric(vertical: 13, horizontal: 16),
            ),
            backgroundColor: backgroundColor != null ? WidgetStatePropertyAll(isDark ? Colors.black : backgroundColor) : color == PrimaryButtonColor.primary
                ? primaryButtonBackground(context)
                : errorButtonBackground(context),
            shape: backgroundColor == null ? null : WidgetStateProperty.all(RoundedRectangleBorder(borderRadius: BorderRadius.circular(8), side:  BorderSide(color: showBorder ? ColorPalette.primary50 : Colors.transparent)))
        ),
        child: isLoading
            ? const SizedBox(
          height: 24,
          width: 24,
          child: CircularProgressIndicator(
            strokeWidth: 2,
            valueColor: AlwaysStoppedAnimation<Color>(Colors.white),
          ),
        )
            : child,
      ),
    );
  }

  WidgetStateProperty<Color> primaryButtonBackground(BuildContext context) =>
      WidgetStateProperty.resolveWith(
        (states) {
          if (states.contains(WidgetState.disabled)) {
            return context.theme.colorScheme.onSurface.withValues(alpha: 0.12);
          } else if (states.contains(WidgetState.hovered)) {
            return context.colorScheme.primary;
          } else if (states.contains(WidgetState.pressed)) {
            return ColorPalette.primary40;
          } else {
            return context.colorScheme.primary;
          }
        },
      );

  WidgetStateProperty<Color> errorButtonBackground(BuildContext context) =>
      WidgetStateProperty.resolveWith(
        (states) {
          if (states.contains(WidgetState.disabled)) {
            return context.theme.colorScheme.onSurface.withValues(alpha: 0.12);
          } else if (states.contains(WidgetState.hovered)) {
            return ColorPalette.error50;
          } else if (states.contains(WidgetState.pressed)) {
            return ColorPalette.error30;
          } else {
            return ColorPalette.error40;
          }
        },
      );
}

enum PrimaryButtonColor { primary, error }
